/**
 * 主应用逻辑
 */

const App = {
    currentPage: 'dashboard',
    currentTaskNo: null,
    
    // 配件汇总表排序状态
    partSummarySort: {
        sortBy: 'total_suggest_amount',
        sortOrder: 'desc'
    },

    // 配件汇总表筛选状态
    partSummaryFilters: {
        page: 1,
        page_size: 50,
        part_code: '',
        priority: ''
    },

    // 侧边栏折叠状态
    isSidebarCollapsed: localStorage.getItem('sidebar-collapsed') === 'true',

    /**
     * 切换侧边栏折叠状态
     */
    toggleSidebarCollapse() {
        this.isSidebarCollapsed = !this.isSidebarCollapsed;
        localStorage.setItem('sidebar-collapsed', this.isSidebarCollapsed);
        this.applySidebarState();
    },

    /**
     * 应用侧边栏状态
     */
    applySidebarState() {
        const sidebar = document.querySelector('.sidebar');
        // 主内容区通过 CSS 选择器 .sidebar.collapsed + ... .main-content 自动调整
        // 或者我们需要手动给 main-content 加类，但 CSS 中用了兄弟选择器，
        // 不过兄弟选择器在这里可能不生效，因为中间隔了 .sidebar-overlay
        // 让我们看看 index.html 结构： sidebar, sidebar-overlay, main-content
        // CSS 选择器是 .sidebar.collapsed + .sidebar-overlay + .main-content
        // 这样是可以的。
        
        if (this.isSidebarCollapsed) {
            sidebar.classList.add('collapsed');
        } else {
            sidebar.classList.remove('collapsed');
        }
        
        // 触发 icon 刷新以确保显示正确（虽然 CSS 旋转已处理）
        lucide.createIcons();
    },

    /**
     * 切换侧边栏显示状态（移动端）
     */
    toggleSidebar(show) {
        const sidebar = document.querySelector('.sidebar');
        const overlay = document.getElementById('sidebar-overlay');
        
        if (show) {
            sidebar.classList.add('visible');
            overlay.classList.add('active');
        } else {
            sidebar.classList.remove('visible');
            overlay.classList.remove('active');
        }
    },

    /**
     * 应用配件筛选
     */
    applyPartFilters() {
        const partCode = document.getElementById('filter-part-code')?.value || '';
        const priorityElement = document.getElementById('filter-priority');
        const priority = priorityElement ? priorityElement.value : '';
        
        this.partSummaryFilters.part_code = partCode;
        this.partSummaryFilters.priority = priority;
        this.partSummaryFilters.page = 1; // 重置到第一页
        
        this.loadPartSummaries();
    },

    /**
     * 重置配件筛选
     */
    resetPartFilters() {
        this.partSummaryFilters.part_code = '';
        this.partSummaryFilters.priority = '';
        this.partSummaryFilters.page = 1;
        
        this.loadPartSummaries();
    },


    /**
     * 渲染分析报告标签页（四大板块：库存概览/销量分析/健康度/补货建议）
     */
    async renderReportTab(container, taskNo) {
        container.innerHTML = '<div class="loading-shops">加载分析报告...</div>';
        
        try {
            const report = await API.getAnalysisReport(taskNo);
            
            if (!report) {
                container.innerHTML = `
                    <div class="card">
                        ${Components.renderEmptyState('file-x', '暂无分析报告', '该任务尚未生成分析报告')}
                    </div>
                `;
                return;
            }

            container.innerHTML = `
                <div id="report-inventory-overview" class="report-module"></div>
                <div id="report-sales-analysis" class="report-module"></div>
                <div id="report-inventory-health" class="report-module"></div>
                <div id="report-replenishment-summary" class="report-module"></div>
            `;

            this.renderInventoryOverview(
                document.getElementById('report-inventory-overview'),
                report.inventory_overview
            );
            this.renderSalesAnalysis(
                document.getElementById('report-sales-analysis'),
                report.sales_analysis
            );
            this.renderInventoryHealth(
                document.getElementById('report-inventory-health'),
                report.inventory_health
            );
            this.renderReplenishmentSummary(
                document.getElementById('report-replenishment-summary'),
                report.replenishment_summary
            );
            
            lucide.createIcons();
        } catch (error) {
            container.innerHTML = `
                <div class="card" style="text-align: center; color: var(--color-danger);">
                    <i data-lucide="alert-circle" style="width: 48px; height: 48px; margin-bottom: 1rem;"></i>
                    <p>加载报告失败: ${error.message}</p>
                </div>
            `;
            lucide.createIcons();
        }
    },

    /**
     * 渲染库存概览板块
     */
    renderInventoryOverview(container, data) {
        if (!data) {
            container.innerHTML = '';
            return;
        }
        const stats = data.stats || {};
        const analysis = data.llm_analysis || {};
        
        // 兼容新旧数据结构
        const conclusion = analysis.conclusion || analysis;
        const process = analysis.analysis_process || null;

        const ratio = stats.overall_ratio;
        const ratioDisplay = (ratio === 999 || ratio === null || ratio === undefined) ? '无销量' : Components.formatNumber(ratio);

        // LLM 分析文本渲染
        let analysisHtml = '';
        if (analysis.error) {
            analysisHtml = `<div class="report-analysis-text"><span style="color:var(--color-warning);">分析生成失败: ${analysis.error}</span></div>`;
        } else {
            const sections = [];
            
            // 季节信息（如果有）
            if (process && process.seasonal_analysis) {
                const sa = process.seasonal_analysis;
                sections.push(`<div class="analysis-block">
                    <div class="analysis-block-title"><i data-lucide="sun"></i> 季节性分析 <span class="season-tag"><i data-lucide="calendar"></i>${sa.current_season || ''}</span></div>
                    <p>${sa.season_demand_feature || ''}</p>
                    <p>${sa.inventory_fitness || ''}</p>
                    ${sa.upcoming_season_preparation ? `<p>下季准备: ${sa.upcoming_season_preparation}</p>` : ''}
                </div>`);
            }
            
            if (conclusion.capital_assessment) {
                const ca = conclusion.capital_assessment;
                sections.push(`<div class="analysis-block">
                    <div class="analysis-block-title"><i data-lucide="wallet"></i> 资金占用评估 <span class="risk-tag risk-tag-${ca.risk_level || 'medium'}">${ca.risk_level === 'high' ? '高风险' : ca.risk_level === 'low' ? '低风险' : '中风险'}</span></div>
                    <p>${ca.total_evaluation || ''}</p>
                    <p>${ca.structure_ratio || ''}</p>
                </div>`);
            }
            if (conclusion.ratio_diagnosis) {
                const rd = conclusion.ratio_diagnosis;
                sections.push(`<div class="analysis-block">
                    <div class="analysis-block-title"><i data-lucide="gauge"></i> 库销比诊断 — ${rd.level || ''}</div>
                    <p>${rd.analysis || ''}</p>
                    <p class="analysis-benchmark">${rd.benchmark || ''}</p>
                </div>`);
            }
            if (conclusion.recommendations && conclusion.recommendations.length > 0) {
                const recHtml = conclusion.recommendations.map(r => {
                    if (typeof r === 'object') {
                        return `<li><strong>${r.action || ''}</strong>${r.reason ? ` - ${r.reason}` : ''}${r.expected_effect ? `<br><small>预期效果: ${r.expected_effect}</small>` : ''}</li>`;
                    }
                    return `<li>${r}</li>`;
                }).join('');
                sections.push(`<div class="analysis-block">
                    <div class="analysis-block-title"><i data-lucide="lightbulb"></i> 库存结构建议</div>
                    <ul class="analysis-rec-list">${recHtml}</ul>
                </div>`);
            }
            
            // 推理过程（可折叠）
            let processHtml = '';
            if (process) {
                processHtml = this._renderAnalysisProcess(process, 'inventory-overview');
            }
            
            analysisHtml = sections.length > 0 ? `<div class="report-analysis-text">${sections.join('')}${processHtml}</div>` : '';
        }

        container.innerHTML = `
            <div class="report-section-title">
                <i data-lucide="warehouse"></i>
                库存总体概览
            </div>
            <div class="report-stat-cards">
                <div class="report-stat-card">
                    <div class="report-stat-label">有效库存总数量</div>
                    <div class="report-stat-value">${Components.formatNumber(stats.total_valid_storage_cnt)}</div>
                </div>
                <div class="report-stat-card">
                    <div class="report-stat-label">资金占用（总金额）</div>
                    <div class="report-stat-value">${Components.formatAmount(stats.total_valid_storage_amount)}</div>
                </div>
                <div class="report-stat-card">
                    <div class="report-stat-label">整体库销比</div>
                    <div class="report-stat-value">${ratioDisplay}</div>
                </div>
                <div class="report-stat-card">
                    <div class="report-stat-label">配件种类数</div>
                    <div class="report-stat-value">${stats.part_count || 0}</div>
                </div>
            </div>
            <div class="report-detail-table">
                <table>
                    <thead>
                        <tr>
                            <th>构成项</th>
                            <th>数量</th>
                            <th>金额</th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td>在库未锁</td>
                            <td>${Components.formatNumber(stats.total_in_stock_unlocked_cnt)}</td>
                            <td>${Components.formatAmount(stats.total_in_stock_unlocked_amount)}</td>
                        </tr>
                        <tr>
                            <td>在途</td>
                            <td>${Components.formatNumber(stats.total_on_the_way_cnt)}</td>
                            <td>${Components.formatAmount(stats.total_on_the_way_amount)}</td>
                        </tr>
                        <tr>
                            <td>计划数</td>
                            <td>${Components.formatNumber(stats.total_has_plan_cnt)}</td>
                            <td>${Components.formatAmount(stats.total_has_plan_amount)}</td>
                        </tr>
                    </tbody>
                </table>
            </div>
            ${analysisHtml}
        `;
        
        // 绑定折叠事件
        this._bindProcessToggle(container);
    },

    /**
     * 渲染销量分析板块
     */
    renderSalesAnalysis(container, data) {
        if (!data) {
            container.innerHTML = '';
            return;
        }
        const stats = data.stats || {};
        const analysis = data.llm_analysis || {};
        
        // 兼容新旧数据结构
        const conclusion = analysis.conclusion || analysis;
        const process = analysis.analysis_process || null;

        // LLM 分析文本
        let analysisHtml = '';
        if (analysis.error) {
            analysisHtml = `<div class="report-analysis-text"><span style="color:var(--color-warning);">分析生成失败: ${analysis.error}</span></div>`;
        } else {
            const sections = [];
            
            // 季节信息（如果有）
            if (process && process.seasonal_analysis) {
                const sa = process.seasonal_analysis;
                sections.push(`<div class="analysis-block">
                    <div class="analysis-block-title"><i data-lucide="sun"></i> 季节性分析 <span class="season-tag"><i data-lucide="calendar"></i>${sa.current_season || ''}</span></div>
                    <p>${sa.expected_performance || ''}</p>
                    <p>${sa.actual_vs_expected || ''}</p>
                    ${sa.seasonal_items_status ? `<p>${sa.seasonal_items_status}</p>` : ''}
                </div>`);
            }
            
            if (conclusion.composition_analysis) {
                const ca = conclusion.composition_analysis;
                sections.push(`<div class="analysis-block">
                    <div class="analysis-block-title"><i data-lucide="pie-chart"></i> 销量构成解读</div>
                    <p>${ca.main_driver || ''}</p>
                    <p>${ca.pending_orders_impact || ''}</p>
                    <p>${ca.booking_trend || ''}</p>
                </div>`);
            }
            if (conclusion.activity_assessment) {
                const aa = conclusion.activity_assessment;
                sections.push(`<div class="analysis-block">
                    <div class="analysis-block-title"><i data-lucide="activity"></i> 销售活跃度</div>
                    <p>${aa.active_ratio || ''}</p>
                    <p>${aa.optimization_suggestion || ''}</p>
                </div>`);
            }
            if (conclusion.demand_trend) {
                const dt = conclusion.demand_trend;
                const dirIcon = dt.direction === '上升' ? 'trending-up' : dt.direction === '下降' ? 'trending-down' : 'minus';
                sections.push(`<div class="analysis-block">
                    <div class="analysis-block-title"><i data-lucide="${dirIcon}"></i> 需求趋势 — ${dt.direction || ''}</div>
                    <p>${dt.evidence || ''}</p>
                    ${dt.seasonal_factor ? `<p>季节因素: ${dt.seasonal_factor}</p>` : ''}
                    <p>${dt.forecast || ''}</p>
                </div>`);
            }
            
            // 推理过程（可折叠）
            let processHtml = '';
            if (process) {
                processHtml = this._renderAnalysisProcess(process, 'sales-analysis');
            }
            
            analysisHtml = sections.length > 0 ? `<div class="report-analysis-text">${sections.join('')}${processHtml}</div>` : '';
        }

        const totalParts = (stats.has_sales_part_count || 0) + (stats.no_sales_part_count || 0);

        container.innerHTML = `
            <div class="report-section-title">
                <i data-lucide="bar-chart-3"></i>
                销量分析
            </div>
            <div class="report-stat-cards">
                <div class="report-stat-card">
                    <div class="report-stat-label">月均销量总数量</div>
                    <div class="report-stat-value">${Components.formatNumber(stats.total_avg_sales_cnt)}</div>
                </div>
                <div class="report-stat-card">
                    <div class="report-stat-label">月均销量总金额</div>
                    <div class="report-stat-value">${Components.formatAmount(stats.total_avg_sales_amount)}</div>
                </div>
                <div class="report-stat-card">
                    <div class="report-stat-label">有销量配件</div>
                    <div class="report-stat-value">${stats.has_sales_part_count || 0} <span class="report-stat-sub">/ ${totalParts}</span></div>
                </div>
                <div class="report-stat-card">
                    <div class="report-stat-label">无销量配件</div>
                    <div class="report-stat-value">${stats.no_sales_part_count || 0} <span class="report-stat-sub">/ ${totalParts}</span></div>
                </div>
            </div>
            <div class="report-detail-table">
                <table>
                    <thead>
                        <tr>
                            <th>构成项</th>
                            <th>总量</th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td>90天出库数</td>
                            <td>${Components.formatNumber(stats.total_out_stock_cnt)}</td>
                        </tr>
                        <tr>
                            <td>未关单已锁</td>
                            <td>${Components.formatNumber(stats.total_storage_locked_cnt)}</td>
                        </tr>
                        <tr>
                            <td>未关单出库</td>
                            <td>${Components.formatNumber(stats.total_out_stock_ongoing_cnt)}</td>
                        </tr>
                        <tr>
                            <td>订件</td>
                            <td>${Components.formatNumber(stats.total_buy_cnt)}</td>
                        </tr>
                    </tbody>
                </table>
            </div>
            ${analysisHtml}
        `;
        
        // 绑定折叠事件
        this._bindProcessToggle(container);
    },

    /**
     * 渲染库存健康度板块（含 Chart.js 环形图）
     */
    renderInventoryHealth(container, data) {
        if (!data) {
            container.innerHTML = '';
            return;
        }
        const stats = data.stats || {};
        const chartData = data.chart_data || {};
        const analysis = data.llm_analysis || {};
        
        // 兼容新旧数据结构
        const conclusion = analysis.conclusion || analysis;
        const process = analysis.analysis_process || null;

        const shortage = stats.shortage || {};
        const stagnant = stats.stagnant || {};
        const low_freq = stats.low_freq || {};
        const normal = stats.normal || {};

        // LLM 分析文本
        let analysisHtml = '';
        if (analysis.error) {
            analysisHtml = `<div class="report-analysis-text"><span style="color:var(--color-warning);">分析生成失败: ${analysis.error}</span></div>`;
        } else {
            const sections = [];
            
            // 季节信息（如果有）
            if (process && process.seasonal_analysis) {
                const sa = process.seasonal_analysis;
                sections.push(`<div class="analysis-block">
                    <div class="analysis-block-title"><i data-lucide="sun"></i> 季节性分析 <span class="season-tag"><i data-lucide="calendar"></i>${sa.current_season || ''}</span></div>
                    ${sa.seasonal_stagnant_items ? `<p>${sa.seasonal_stagnant_items}</p>` : ''}
                    ${sa.seasonal_shortage_risk ? `<p>${sa.seasonal_shortage_risk}</p>` : ''}
                    ${sa.upcoming_season_alert ? `<p>下季关注: ${sa.upcoming_season_alert}</p>` : ''}
                </div>`);
            }
            
            if (conclusion.health_score) {
                const hs = conclusion.health_score;
                sections.push(`<div class="analysis-block">
                    <div class="analysis-block-title"><i data-lucide="heart-pulse"></i> 健康度评分 — ${hs.score || ''}</div>
                    <p>${hs.normal_ratio_evaluation || ''}</p>
                </div>`);
            }
            if (conclusion.problem_diagnosis) {
                const pd = conclusion.problem_diagnosis;
                sections.push(`<div class="analysis-block">
                    <div class="analysis-block-title"><i data-lucide="stethoscope"></i> 问题诊断</div>
                    ${pd.stagnant_analysis ? `<p>呆滞件: ${pd.stagnant_analysis}</p>` : ''}
                    ${pd.shortage_analysis ? `<p>缺货件: ${pd.shortage_analysis}</p>` : ''}
                    ${pd.low_freq_analysis ? `<p>低频件: ${pd.low_freq_analysis}</p>` : ''}
                </div>`);
            }
            if (conclusion.capital_release) {
                const cr = conclusion.capital_release;
                sections.push(`<div class="analysis-block">
                    <div class="analysis-block-title"><i data-lucide="banknote"></i> 资金释放机会</div>
                    ${cr.stagnant_releasable ? `<p>呆滞件可释放: ${cr.stagnant_releasable}</p>` : ''}
                    ${cr.low_freq_releasable ? `<p>低频件可释放: ${cr.low_freq_releasable}</p>` : ''}
                    ${cr.action_plan ? `<p>${cr.action_plan}</p>` : ''}
                </div>`);
            }
            if (conclusion.priority_actions && conclusion.priority_actions.length > 0) {
                const actHtml = conclusion.priority_actions.map(a => {
                    if (typeof a === 'object') {
                        return `<li><strong>${a.action || ''}</strong>${a.reason ? ` - ${a.reason}` : ''}${a.expected_effect ? `<br><small>预期效果: ${a.expected_effect}</small>` : ''}</li>`;
                    }
                    return `<li>${a}</li>`;
                }).join('');
                sections.push(`<div class="analysis-block">
                    <div class="analysis-block-title"><i data-lucide="list-ordered"></i> 改善优先级</div>
                    <ol class="analysis-rec-list">${actHtml}</ol>
                </div>`);
            }
            
            // 推理过程（可折叠）
            let processHtml = '';
            if (process) {
                processHtml = this._renderAnalysisProcess(process, 'inventory-health');
            }
            
            analysisHtml = sections.length > 0 ? `<div class="report-analysis-text">${sections.join('')}${processHtml}</div>` : '';
        }

        container.innerHTML = `
            <div class="report-section-title">
                <i data-lucide="heart-pulse"></i>
                库存构成健康度
            </div>
            <div class="report-stat-cards report-stat-cards-4">
                <div class="report-stat-card report-stat-card-danger">
                    <div class="report-stat-label">缺货件</div>
                    <div class="report-stat-value">${shortage.count || 0}</div>
                    <div class="report-stat-pct">${Components.formatNumber(shortage.count_pct)}% · ${Components.formatAmount(shortage.amount)}</div>
                </div>
                <div class="report-stat-card report-stat-card-warning">
                    <div class="report-stat-label">呆滞件</div>
                    <div class="report-stat-value">${stagnant.count || 0}</div>
                    <div class="report-stat-pct">${Components.formatNumber(stagnant.count_pct)}% · ${Components.formatAmount(stagnant.amount)}</div>
                </div>
                <div class="report-stat-card report-stat-card-info">
                    <div class="report-stat-label">低频件</div>
                    <div class="report-stat-value">${low_freq.count || 0}</div>
                    <div class="report-stat-pct">${Components.formatNumber(low_freq.count_pct)}% · ${Components.formatAmount(low_freq.amount)}</div>
                </div>
                <div class="report-stat-card report-stat-card-success">
                    <div class="report-stat-label">正常件</div>
                    <div class="report-stat-value">${normal.count || 0}</div>
                    <div class="report-stat-pct">${Components.formatNumber(normal.count_pct)}% · ${Components.formatAmount(normal.amount)}</div>
                </div>
            </div>
            <div class="report-charts-row">
                <div class="report-chart-container">
                    <div class="report-chart-title">数量占比</div>
                    <canvas id="health-count-chart"></canvas>
                </div>
                <div class="report-chart-container">
                    <div class="report-chart-title">金额占比</div>
                    <canvas id="health-amount-chart"></canvas>
                </div>
            </div>
            ${analysisHtml}
        `;

        // 渲染 Chart.js 环形图
        this._renderHealthCharts(chartData);
        
        // 绑定折叠事件
        this._bindProcessToggle(container);
    },

    /**
     * 渲染健康度环形图
     */
    _renderHealthCharts(chartData) {
        if (!chartData || !chartData.labels) return;
        if (typeof Chart === 'undefined') return;

        const colors = ['#ef4444', '#f59e0b', '#3b82f6', '#10b981'];
        const borderColors = ['#dc2626', '#d97706', '#2563eb', '#059669'];

        const chartOptions = {
            responsive: true,
            maintainAspectRatio: true,
            plugins: {
                legend: {
                    position: 'bottom',
                    labels: {
                        color: '#94a3b8',
                        padding: 16,
                        usePointStyle: true,
                        pointStyleWidth: 10,
                        font: { size: 12 }
                    }
                },
                tooltip: {
                    backgroundColor: '#1e293b',
                    titleColor: '#f8fafc',
                    bodyColor: '#94a3b8',
                    borderColor: 'rgba(148,163,184,0.2)',
                    borderWidth: 1
                }
            },
            cutout: '60%'
        };

        // 数量占比图
        const countCtx = document.getElementById('health-count-chart');
        if (countCtx) {
            new Chart(countCtx, {
                type: 'doughnut',
                data: {
                    labels: chartData.labels,
                    datasets: [{
                        data: chartData.count_values,
                        backgroundColor: colors,
                        borderColor: borderColors,
                        borderWidth: 2
                    }]
                },
                options: chartOptions
            });
        }

        // 金额占比图
        const amountCtx = document.getElementById('health-amount-chart');
        if (amountCtx) {
            new Chart(amountCtx, {
                type: 'doughnut',
                data: {
                    labels: chartData.labels,
                    datasets: [{
                        data: chartData.amount_values,
                        backgroundColor: colors,
                        borderColor: borderColors,
                        borderWidth: 2
                    }]
                },
                options: {
                    ...chartOptions,
                    plugins: {
                        ...chartOptions.plugins,
                        tooltip: {
                            ...chartOptions.plugins.tooltip,
                            callbacks: {
                                label: function(context) {
                                    const value = context.parsed;
                                    return ` ${context.label}: ¥${Number(value).toLocaleString('zh-CN', {minimumFractionDigits: 2})}`;
                                }
                            }
                        }
                    }
                }
            });
        }
    },

    /**
     * 渲染补货建议板块
     */
    renderReplenishmentSummary(container, data) {
        if (!data) {
            container.innerHTML = '';
            return;
        }
        const stats = data.stats || {};
        const analysis = data.llm_analysis || {};
        
        // 兼容新旧数据结构
        const conclusion = analysis.conclusion || analysis;
        const process = analysis.analysis_process || null;

        const urgent = stats.urgent || {};
        const suggested = stats.suggested || {};
        const optional = stats.optional || {};

        // LLM 分析文本
        let analysisHtml = '';
        if (analysis.error) {
            analysisHtml = `<div class="report-analysis-text"><span style="color:var(--color-warning);">分析生成失败: ${analysis.error}</span></div>`;
        } else {
            const sections = [];
            
            // 季节信息（如果有）
            if (process && process.seasonal_analysis) {
                const sa = process.seasonal_analysis;
                sections.push(`<div class="analysis-block">
                    <div class="analysis-block-title"><i data-lucide="sun"></i> 季节性分析 <span class="season-tag"><i data-lucide="calendar"></i>${sa.current_season || ''}</span></div>
                    ${sa.seasonal_priority_items ? `<p>${sa.seasonal_priority_items}</p>` : ''}
                    ${sa.timeline_adjustment ? `<p>${sa.timeline_adjustment}</p>` : ''}
                    ${sa.next_season_preparation ? `<p>下季准备: ${sa.next_season_preparation}</p>` : ''}
                </div>`);
            }
            
            if (conclusion.urgency_assessment) {
                const ua = conclusion.urgency_assessment;
                const riskTag = ua.risk_level === 'high' ? '高风险' : ua.risk_level === 'low' ? '低风险' : '中风险';
                sections.push(`<div class="analysis-block">
                    <div class="analysis-block-title"><i data-lucide="alert-triangle"></i> 紧迫度评估 <span class="risk-tag risk-tag-${ua.risk_level || 'medium'}">${riskTag}</span></div>
                    <p>${ua.urgent_ratio_evaluation || ''}</p>
                    ${ua.immediate_action_needed ? '<p style="color:var(--color-danger);font-weight:600;">需要立即采取行动</p>' : ''}
                </div>`);
            }
            if (conclusion.budget_allocation) {
                const ba = conclusion.budget_allocation;
                sections.push(`<div class="analysis-block">
                    <div class="analysis-block-title"><i data-lucide="wallet"></i> 资金分配建议</div>
                    <p>${ba.recommended_order || ''}</p>
                    ${ba.urgent_budget ? `<p>急需补货预算: ${ba.urgent_budget}</p>` : ''}
                    ${ba.suggested_budget ? `<p>建议补货预算: ${ba.suggested_budget}</p>` : ''}
                    ${ba.optional_budget ? `<p>可选补货预算: ${ba.optional_budget}</p>` : ''}
                </div>`);
            }
            if (conclusion.execution_plan) {
                const ep = conclusion.execution_plan;
                sections.push(`<div class="analysis-block">
                    <div class="analysis-block-title"><i data-lucide="calendar-clock"></i> 执行节奏建议</div>
                    ${ep.urgent_timeline ? `<p>急需: ${ep.urgent_timeline}</p>` : ''}
                    ${ep.suggested_timeline ? `<p>建议: ${ep.suggested_timeline}</p>` : ''}
                    ${ep.optional_timeline ? `<p>可选: ${ep.optional_timeline}</p>` : ''}
                </div>`);
            }
            if (conclusion.risk_warnings && conclusion.risk_warnings.length > 0) {
                const warnHtml = conclusion.risk_warnings.map(w => {
                    if (typeof w === 'object') {
                        return `<li><strong>${w.risk_type || ''}</strong>: ${w.description || ''}${w.mitigation ? `<br><small>应对: ${w.mitigation}</small>` : ''}</li>`;
                    }
                    return `<li>${w}</li>`;
                }).join('');
                sections.push(`<div class="analysis-block">
                    <div class="analysis-block-title"><i data-lucide="shield-alert"></i> 风险提示</div>
                    <ul class="analysis-rec-list">${warnHtml}</ul>
                </div>`);
            }
            
            // 推理过程（可折叠）
            let processHtml = '';
            if (process) {
                processHtml = this._renderAnalysisProcess(process, 'replenishment-summary');
            }
            
            analysisHtml = sections.length > 0 ? `<div class="report-analysis-text">${sections.join('')}${processHtml}</div>` : '';
        }

        container.innerHTML = `
            <div class="report-section-title">
                <i data-lucide="shopping-cart"></i>
                补货建议生成情况
            </div>
            <div class="report-detail-table">
                <table>
                    <thead>
                        <tr>
                            <th>优先级</th>
                            <th>配件种类数</th>
                            <th>建议补货金额</th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td><span class="priority-badge priority-high">急需补货</span></td>
                            <td>${urgent.count || 0}</td>
                            <td class="table-cell-amount">${Components.formatAmount(urgent.amount)}</td>
                        </tr>
                        <tr>
                            <td><span class="priority-badge priority-medium">建议补货</span></td>
                            <td>${suggested.count || 0}</td>
                            <td class="table-cell-amount">${Components.formatAmount(suggested.amount)}</td>
                        </tr>
                        <tr>
                            <td><span class="priority-badge priority-low">可选补货</span></td>
                            <td>${optional.count || 0}</td>
                            <td class="table-cell-amount">${Components.formatAmount(optional.amount)}</td>
                        </tr>
                        <tr style="font-weight:600; border-top: 2px solid var(--border-color-light);">
                            <td>合计</td>
                            <td>${stats.total_count || 0}</td>
                            <td class="table-cell-amount">${Components.formatAmount(stats.total_amount)}</td>
                        </tr>
                    </tbody>
                </table>
            </div>
            ${analysisHtml}
        `;
        
        // 绑定折叠事件
        this._bindProcessToggle(container);
    },

    /**
     * 渲染推理过程（可折叠）
     */
    _renderAnalysisProcess(process, moduleId) {
        if (!process) return '';
        
        const sections = [];
        
        // 计算指标
        if (process.calculated_metrics) {
            const items = Object.entries(process.calculated_metrics)
                .filter(([k, v]) => v && v !== '')
                .map(([k, v]) => `<div class="process-item"><span class="process-item-label">${this._formatProcessKey(k)}</span><span class="process-item-value">${v}</span></div>`)
                .join('');
            if (items) {
                sections.push(`<div class="process-section"><div class="process-section-title">计算指标</div>${items}</div>`);
            }
        }
        
        // 库销比诊断
        if (process.ratio_diagnosis) {
            const rd = process.ratio_diagnosis;
            const items = [];
            if (rd.current_value) items.push(`<div class="process-item"><span class="process-item-label">当前值</span><span class="process-item-value highlight">${rd.current_value}</span></div>`);
            if (rd.level) items.push(`<div class="process-item"><span class="process-item-label">判断等级</span><span class="process-item-value highlight">${rd.level}</span></div>`);
            if (rd.reasoning) items.push(`<div class="process-item"><span class="process-item-label">判断依据</span><span class="process-item-value">${rd.reasoning}</span></div>`);
            if (rd.benchmark_comparison) items.push(`<div class="process-item"><span class="process-item-label">行业对比</span><span class="process-item-value">${rd.benchmark_comparison}</span></div>`);
            if (items.length > 0) {
                sections.push(`<div class="process-section"><div class="process-section-title">库销比诊断</div>${items.join('')}</div>`);
            }
        }
        
        // 结构分析
        if (process.structure_analysis) {
            const sa = process.structure_analysis;
            const items = [];
            if (sa.in_stock_evaluation) items.push(`<div class="process-item"><span class="process-item-label">在库未锁</span><span class="process-item-value">${sa.in_stock_evaluation}</span></div>`);
            if (sa.on_way_evaluation) items.push(`<div class="process-item"><span class="process-item-label">在途</span><span class="process-item-value">${sa.on_way_evaluation}</span></div>`);
            if (sa.plan_evaluation) items.push(`<div class="process-item"><span class="process-item-label">计划数</span><span class="process-item-value">${sa.plan_evaluation}</span></div>`);
            if (sa.abnormal_items && sa.abnormal_items.length > 0) {
                items.push(`<div class="process-item"><span class="process-item-label">异常项</span><span class="process-item-value">${sa.abnormal_items.join('; ')}</span></div>`);
            }
            if (items.length > 0) {
                sections.push(`<div class="process-section"><div class="process-section-title">结构分析</div>${items.join('')}</div>`);
            }
        }
        
        // 构成诊断（销量分析）
        if (process.composition_diagnosis) {
            const cd = process.composition_diagnosis;
            const items = [];
            if (cd.out_stock_evaluation) items.push(`<div class="process-item"><span class="process-item-label">90天出库</span><span class="process-item-value">${cd.out_stock_evaluation}</span></div>`);
            if (cd.locked_evaluation) items.push(`<div class="process-item"><span class="process-item-label">未关单已锁</span><span class="process-item-value">${cd.locked_evaluation}</span></div>`);
            if (cd.ongoing_evaluation) items.push(`<div class="process-item"><span class="process-item-label">未关单出库</span><span class="process-item-value">${cd.ongoing_evaluation}</span></div>`);
            if (cd.buy_evaluation) items.push(`<div class="process-item"><span class="process-item-label">订件</span><span class="process-item-value">${cd.buy_evaluation}</span></div>`);
            if (items.length > 0) {
                sections.push(`<div class="process-section"><div class="process-section-title">构成诊断</div>${items.join('')}</div>`);
            }
        }
        
        // 活跃度诊断
        if (process.activity_diagnosis) {
            const ad = process.activity_diagnosis;
            const items = [];
            if (ad.current_rate) items.push(`<div class="process-item"><span class="process-item-label">当前活跃率</span><span class="process-item-value highlight">${ad.current_rate}</span></div>`);
            if (ad.level) items.push(`<div class="process-item"><span class="process-item-label">判断等级</span><span class="process-item-value highlight">${ad.level}</span></div>`);
            if (ad.reasoning) items.push(`<div class="process-item"><span class="process-item-label">判断依据</span><span class="process-item-value">${ad.reasoning}</span></div>`);
            if (items.length > 0) {
                sections.push(`<div class="process-section"><div class="process-section-title">活跃度诊断</div>${items.join('')}</div>`);
            }
        }
        
        // 趋势诊断
        if (process.trend_diagnosis) {
            const td = process.trend_diagnosis;
            const items = [];
            if (td.signals && td.signals.length > 0) items.push(`<div class="process-item"><span class="process-item-label">趋势信号</span><span class="process-item-value">${td.signals.join('; ')}</span></div>`);
            if (td.reasoning) items.push(`<div class="process-item"><span class="process-item-label">判断依据</span><span class="process-item-value">${td.reasoning}</span></div>`);
            if (items.length > 0) {
                sections.push(`<div class="process-section"><div class="process-section-title">趋势诊断</div>${items.join('')}</div>`);
            }
        }
        
        // 健康度诊断
        if (process.health_score_diagnosis) {
            const hsd = process.health_score_diagnosis;
            const items = [];
            if (hsd.normal_ratio) items.push(`<div class="process-item"><span class="process-item-label">正常件占比</span><span class="process-item-value highlight">${hsd.normal_ratio}</span></div>`);
            if (hsd.score) items.push(`<div class="process-item"><span class="process-item-label">健康度评分</span><span class="process-item-value highlight">${hsd.score}</span></div>`);
            if (hsd.reasoning) items.push(`<div class="process-item"><span class="process-item-label">判断依据</span><span class="process-item-value">${hsd.reasoning}</span></div>`);
            if (items.length > 0) {
                sections.push(`<div class="process-section"><div class="process-section-title">健康度诊断</div>${items.join('')}</div>`);
            }
        }
        
        // 问题诊断（健康度）
        if (process.problem_diagnosis) {
            const pd = process.problem_diagnosis;
            const items = [];
            ['shortage', 'stagnant', 'low_freq'].forEach(key => {
                const item = pd[key];
                if (item) {
                    const label = key === 'shortage' ? '缺货件' : key === 'stagnant' ? '呆滞件' : '低频件';
                    if (item.threshold_comparison) items.push(`<div class="process-item"><span class="process-item-label">${label}</span><span class="process-item-value">${item.threshold_comparison}</span></div>`);
                }
            });
            if (items.length > 0) {
                sections.push(`<div class="process-section"><div class="process-section-title">问题诊断</div>${items.join('')}</div>`);
            }
        }
        
        // 资金释放计算
        if (process.capital_release_calculation) {
            const crc = process.capital_release_calculation;
            const items = [];
            if (crc.stagnant_calculation) items.push(`<div class="process-item"><span class="process-item-label">呆滞件</span><span class="process-item-value">${crc.stagnant_calculation}</span></div>`);
            if (crc.low_freq_calculation) items.push(`<div class="process-item"><span class="process-item-label">低频件</span><span class="process-item-value">${crc.low_freq_calculation}</span></div>`);
            if (crc.total_releasable) items.push(`<div class="process-item"><span class="process-item-label">总可释放</span><span class="process-item-value highlight">${crc.total_releasable}</span></div>`);
            if (items.length > 0) {
                sections.push(`<div class="process-section"><div class="process-section-title">资金释放计算</div>${items.join('')}</div>`);
            }
        }
        
        // 紧迫度诊断（补货建议）
        if (process.urgency_diagnosis) {
            const ud = process.urgency_diagnosis;
            const items = [];
            if (ud.urgent_ratio) items.push(`<div class="process-item"><span class="process-item-label">急需占比</span><span class="process-item-value highlight">${ud.urgent_ratio}</span></div>`);
            if (ud.level) items.push(`<div class="process-item"><span class="process-item-label">紧迫等级</span><span class="process-item-value highlight">${ud.level}</span></div>`);
            if (ud.reasoning) items.push(`<div class="process-item"><span class="process-item-label">判断依据</span><span class="process-item-value">${ud.reasoning}</span></div>`);
            if (items.length > 0) {
                sections.push(`<div class="process-section"><div class="process-section-title">紧迫度诊断</div>${items.join('')}</div>`);
            }
        }
        
        // 预算分析
        if (process.budget_analysis) {
            const ba = process.budget_analysis;
            const items = [];
            if (ba.current_distribution) items.push(`<div class="process-item"><span class="process-item-label">当前分布</span><span class="process-item-value">${ba.current_distribution}</span></div>`);
            if (ba.comparison_with_standard) items.push(`<div class="process-item"><span class="process-item-label">标准对比</span><span class="process-item-value">${ba.comparison_with_standard}</span></div>`);
            if (ba.adjustment_needed) items.push(`<div class="process-item"><span class="process-item-label">调整建议</span><span class="process-item-value">${ba.adjustment_needed}</span></div>`);
            if (items.length > 0) {
                sections.push(`<div class="process-section"><div class="process-section-title">预算分析</div>${items.join('')}</div>`);
            }
        }
        
        // 风险识别
        if (process.risk_identification) {
            const ri = process.risk_identification;
            const items = [];
            if (ri.capital_pressure_check) items.push(`<div class="process-item"><span class="process-item-label">资金压力</span><span class="process-item-value">${ri.capital_pressure_check}</span></div>`);
            if (ri.over_replenishment_check) items.push(`<div class="process-item"><span class="process-item-label">过度补货</span><span class="process-item-value">${ri.over_replenishment_check}</span></div>`);
            if (ri.identified_risks && ri.identified_risks.length > 0) {
                items.push(`<div class="process-item"><span class="process-item-label">识别风险</span><span class="process-item-value">${ri.identified_risks.join('; ')}</span></div>`);
            }
            if (items.length > 0) {
                sections.push(`<div class="process-section"><div class="process-section-title">风险识别</div>${items.join('')}</div>`);
            }
        }
        
        if (sections.length === 0) return '';
        
        return `
            <div class="analysis-process-toggle" data-target="${moduleId}-process">
                <i data-lucide="chevron-down"></i>
                <span>查看分析推理过程</span>
            </div>
            <div class="analysis-process-content" id="${moduleId}-process">
                ${sections.join('')}
            </div>
        `;
    },
    
    /**
     * 格式化推理过程的key名称
     */
    _formatProcessKey(key) {
        const keyMap = {
            'in_stock_ratio': '在库未锁占比',
            'on_way_ratio': '在途占比',
            'plan_ratio': '计划数占比',
            'avg_cost': '平均成本',
            'out_stock_ratio': '90天出库占比',
            'locked_ratio': '未关单已锁占比',
            'ongoing_ratio': '未关单出库占比',
            'buy_ratio': '订件占比',
            'sku_active_rate': 'SKU活跃率',
            'avg_sales_price': '平均销售金额',
            'urgent_count_ratio': '急需数量占比',
            'urgent_amount_ratio': '急需金额占比',
            'suggested_count_ratio': '建议数量占比',
            'suggested_amount_ratio': '建议金额占比',
            'optional_count_ratio': '可选数量占比',
            'optional_amount_ratio': '可选金额占比',
        };
        return keyMap[key] || key;
    },
    
    /**
     * 绑定推理过程折叠事件
     */
    _bindProcessToggle(container) {
        const toggles = container.querySelectorAll('.analysis-process-toggle');
        toggles.forEach(toggle => {
            toggle.addEventListener('click', () => {
                const targetId = toggle.dataset.target;
                const content = document.getElementById(targetId);
                if (content) {
                    toggle.classList.toggle('expanded');
                    content.classList.toggle('expanded');
                    lucide.createIcons();
                }
            });
        });
    },

    /**
     * 初始化应用
     */
    init() {
        this.bindEvents();
        this.handleRoute();
        this.applySidebarState(); // 初始化侧边栏状态
        window.addEventListener('hashchange', () => this.handleRoute());
        lucide.createIcons();
    },

    /**
     * 绑定全局事件
     */
    bindEvents() {
        // 刷新按钮
        document.getElementById('refresh-btn').addEventListener('click', () => {
            this.handleRoute();
        });

        // 模态框关闭
        document.getElementById('modal-close').addEventListener('click', () => {
            Components.closeModal();
        });
        document.getElementById('modal-overlay').addEventListener('click', (e) => {
            if (e.target.id === 'modal-overlay') {
                Components.closeModal();
            }
        });

        // 侧边栏切换
        const menuToggle = document.getElementById('menu-toggle');
        const sidebarOverlay = document.getElementById('sidebar-overlay');
        
        if (menuToggle) {
            menuToggle.addEventListener('click', () => {
                this.toggleSidebar(true);
            });
        }
        
        if (sidebarOverlay) {
            sidebarOverlay.addEventListener('click', () => {
                this.toggleSidebar(false);
            });
        }

        // 桌面端侧边栏折叠按钮
        const collapseBtn = document.getElementById('sidebar-collapse-btn');
        if (collapseBtn) {
            collapseBtn.addEventListener('click', () => {
                this.toggleSidebarCollapse();
            });
        }

        // 导航点击自动关闭侧边栏（移动端）
        document.querySelectorAll('.nav-item').forEach(item => {
            item.addEventListener('click', () => {
                if (window.innerWidth <= 1024) {
                    this.toggleSidebar(false);
                }
            });
        });
    },

    /**
     * 路由处理
     */
    handleRoute() {
        const hash = window.location.hash || '#/';
        const [, path, param] = hash.match(/#\/([^/]*)(?:\/(.*))?/) || [, '', ''];

        // 更新导航状态
        document.querySelectorAll('.nav-item').forEach(item => {
            item.classList.remove('active');
            if (item.dataset.page === (path || 'dashboard')) {
                item.classList.add('active');
            }
        });

        // 路由分发
        switch (path) {
            case 'tasks':
                if (param) {
                    this.showTaskDetail(param);
                } else {
                    this.showTaskList();
                }
                break;
            case '':
            default:
                this.showDashboard();
                break;
        }
    },

    /**
     * 更新面包屑
     */
    updateBreadcrumb(items) {
        const breadcrumb = document.getElementById('breadcrumb');
        breadcrumb.innerHTML = items.map((item, index) => {
            if (item.href) {
                return `<span class="breadcrumb-item"><a href="${item.href}">${item.text}</a></span>`;
            }
            return `<span class="breadcrumb-item">${item.text}</span>`;
        }).join('');
    },

    /**
     * 显示概览页面
     */
    async showDashboard() {
        this.currentPage = 'dashboard';
        this.updateBreadcrumb([{ text: '概览' }]);

        const container = document.getElementById('page-container');
        container.innerHTML = '<div class="stats-grid" id="stats-grid"></div><div id="recent-tasks"></div>';

        try {
            // 获取统计数据
            const [stats, tasksData] = await Promise.all([
                API.getStatsSummary().catch(() => ({})),
                API.getTasks({ page: 1, page_size: 5 }).catch(() => ({ items: [] })),
            ]);

            // 渲染统计卡片
            const statsGrid = document.getElementById('stats-grid');
            statsGrid.innerHTML = `
                ${Components.renderStatCard('list-checks', '总任务数', stats.total_tasks || 0, 'primary')}
                ${Components.renderStatCard('check-circle', '成功任务', stats.success_tasks || 0, 'success')}
                ${Components.renderStatCard('x-circle', '失败任务', stats.failed_tasks || 0, 'danger')}
                ${Components.renderStatCard('package', '建议配件', stats.total_parts || 0, 'info')}
                ${Components.renderStatCard('dollar-sign', '建议金额', Components.formatAmount(stats.total_suggest_amount), 'warning')}
            `;

            // 渲染最近任务
            this.renderRecentTasks(tasksData.items || []);

            lucide.createIcons();
        } catch (error) {
            Components.showToast('加载数据失败: ' + error.message, 'error');
        }
    },

    /**
     * 渲染最近任务
     */
    renderRecentTasks(tasks) {
        const container = document.getElementById('recent-tasks');
        
        if (!tasks.length) {
            container.innerHTML = `
                <div class="card">
                    <div class="card-header">
                        <h3 class="card-title">
                            <i data-lucide="clock"></i>
                            最近任务
                        </h3>
                    </div>
                    ${Components.renderEmptyState('inbox', '暂无任务', '还没有执行过任何补货建议任务')}
                </div>
            `;
            return;
        }

        container.innerHTML = `
            <div class="table-container">
                <div class="table-header">
                    <h3 class="table-title">最近任务</h3>
                    <a href="#/tasks" class="btn btn-secondary btn-sm">
                        查看全部
                        <i data-lucide="arrow-right"></i>
                    </a>
                </div>
                <div class="table-wrapper">
                    <table>
                        <thead>
                            <tr>
                                <th>任务编号</th>
                                <th>商家组合</th>
                                <th>状态</th>
                                <th>配件数</th>
                                <th>建议金额</th>
                                <th>执行时间</th>
                            </tr>
                        </thead>
                        <tbody>
                            ${tasks.map(task => `
                                <tr>
                                    <td>
                                        <a href="#/tasks/${task.task_no}" class="table-cell-link table-cell-mono">
                                            ${task.task_no}
                                        </a>
                                    </td>
                                    <td>${task.dealer_grouping_name || '-'}</td>
                                    <td>${Components.getStatusBadge(task.status, task.status_text)}</td>
                                    <td>${task.part_count}</td>
                                    <td class="table-cell-amount">${Components.formatAmount(task.actual_amount)}</td>
                                    <td class="table-cell-secondary">${Components.formatDuration(task.duration_seconds)}</td>
                                </tr>
                            `).join('')}
                        </tbody>
                    </table>
                </div>
            </div>
        `;
    },

    /**
     * 显示任务列表页面
     */
    async showTaskList(page = 1) {
        this.currentPage = 'tasks';
        this.updateBreadcrumb([{ text: '任务列表' }]);

        const container = document.getElementById('page-container');
        container.innerHTML = '<div id="task-list-container"></div>';

        try {
            Components.showLoading();
            const data = await API.getTasks({ page, page_size: 20 });
            Components.hideLoading();

            this.renderTaskList(data);
            lucide.createIcons();
        } catch (error) {
            Components.hideLoading();
            Components.showToast('加载任务列表失败: ' + error.message, 'error');
        }
    },

    /**
     * 渲染任务列表
     */
    renderTaskList(data) {
        const container = document.getElementById('task-list-container');
        const { items, total, page, page_size } = data;

        if (!items.length) {
            container.innerHTML = `
                <div class="card">
                    ${Components.renderEmptyState('inbox', '暂无任务', '还没有执行过任何补货建议任务')}
                </div>
            `;
            return;
        }

        container.innerHTML = `
            <div class="table-container">
                <div class="table-header">
                    <h3 class="table-title">任务列表 (${total})</h3>
                </div>
                <div class="table-wrapper">
                    <table>
                        <thead>
                            <tr>
                                <th>任务编号</th>
                                <th>商家组合</th>
                                <th>状态</th>
                                <th>配件数</th>
                                <th>建议金额</th>
                                <th>基准库销比</th>
                                <th>统计日期</th>
                                <th>执行时长</th>
                                <th>操作</th>
                            </tr>
                        </thead>
                        <tbody>
                            ${items.map(task => `
                                <tr>
                                    <td>
                                        <span class="table-cell-mono">${task.task_no}</span>
                                    </td>
                                    <td>${task.dealer_grouping_name || '-'}</td>
                                    <td>${Components.getStatusBadge(task.status, task.status_text)}</td>
                                    <td>${task.part_count}</td>
                                    <td class="table-cell-amount">${Components.formatAmount(task.actual_amount)}</td>
                                    <td>${Components.formatRatio(task.base_ratio)}</td>
                                    <td>${task.statistics_date || '-'}</td>
                                    <td class="table-cell-secondary">${Components.formatDuration(task.duration_seconds)}</td>
                                    <td>
                                        <a href="#/tasks/${task.task_no}" class="btn btn-ghost btn-sm">
                                            <i data-lucide="eye"></i>
                                            查看
                                        </a>
                                    </td>
                                </tr>
                            `).join('')}
                        </tbody>
                    </table>
                </div>
                <div id="pagination-container"></div>
            </div>
        `;

        // 渲染分页
        const paginationContainer = document.getElementById('pagination-container');
        paginationContainer.innerHTML = Components.renderPagination(page, total, page_size);

        // 绑定分页事件
        paginationContainer.querySelectorAll('.pagination-btn[data-page]').forEach(btn => {
            btn.addEventListener('click', () => {
                const targetPage = parseInt(btn.dataset.page);
                if (targetPage && targetPage !== page) {
                    this.showTaskList(targetPage);
                }
            });
        });

        lucide.createIcons();
    },

    /**
     * 显示任务详情页面
     */
    async showTaskDetail(taskNo) {
        this.currentPage = 'task-detail';
        this.currentTaskNo = taskNo;
        this.updateBreadcrumb([
            { text: '任务列表', href: '#/tasks' },
            { text: taskNo },
        ]);

        const container = document.getElementById('page-container');
        container.innerHTML = '<div id="task-detail-container"></div>';

        try {
            Components.showLoading();
            
            const [task, partSummaries, logs] = await Promise.all([
                API.getTask(taskNo),
                API.getPartSummaries(taskNo, { page: 1, page_size: 100 }).catch(() => ({ items: [], total: 0 })),
                API.getTaskLogs(taskNo).catch(() => ({ items: [] })),
            ]);
            
            Components.hideLoading();
            this.renderTaskDetail(task, partSummaries, logs);
            lucide.createIcons();
        } catch (error) {
            Components.hideLoading();
            Components.showToast('加载任务详情失败: ' + error.message, 'error');
        }
    },

    /**
     * 渲染任务详情
     */
    renderTaskDetail(task, partSummaries, logs) {
        this._currentLogs = logs;
        this._partSummaries = partSummaries;
        const container = document.getElementById('task-detail-container');

        container.innerHTML = `
            <!-- 返回链接 -->
            <a href="#/tasks" class="back-link">
                <i data-lucide="arrow-left"></i>
                返回任务列表
            </a>

            <!-- 任务头部 -->
            <div class="detail-header">
                <div>
                    <h1 class="detail-title">
                        ${task.task_no}
                        ${Components.getStatusBadge(task.status, task.status_text)}
                    </h1>
                    <div class="detail-meta">
                        <span class="detail-meta-item">
                            <i data-lucide="building-2"></i>
                            ${task.dealer_grouping_name || '未知商家组合'}
                        </span>
                        <span class="detail-meta-item">
                            <i data-lucide="calendar"></i>
                            ${task.statistics_date || '-'}
                        </span>
                        <span class="detail-meta-item">
                            <i data-lucide="clock"></i>
                            ${Components.formatDuration(task.duration_seconds)}
                        </span>
                    </div>
                </div>
            </div>

            <!-- 统计卡片 -->
            <div class="stats-grid">
                ${Components.renderStatCard('package', '建议配件数', task.part_count, 'primary')}
                ${Components.renderStatCard('dollar-sign', '建议金额', Components.formatAmount(task.actual_amount), 'success')}
                ${Components.renderStatCard('percent', '基准库销比', Components.formatRatio(task.base_ratio), 'info')}
                ${Components.renderStatCard('cpu', 'LLM Tokens', task.llm_total_tokens || 0, 'warning')}
            </div>

            <!-- 标签页 -->
            <div class="tabs" id="detail-tabs">
                <button class="tab active" data-tab="details">
                    <i data-lucide="list"></i>
                    配件明细
                </button>
                <button class="tab" data-tab="report">
                    <i data-lucide="file-text"></i>
                    分析报告
                </button>
                <button class="tab" data-tab="logs">
                    <i data-lucide="activity"></i>
                    执行日志
                </button>
                <button class="tab" data-tab="info">
                    <i data-lucide="info"></i>
                    任务信息
                </button>
            </div>

            <!-- 标签页内容 -->
            <div id="tab-content" class="report-container"></div>
        `;

        // 绑定标签页事件
        const tabs = container.querySelectorAll('.tab');
        tabs.forEach(tab => {
            tab.addEventListener('click', () => {
                tabs.forEach(t => t.classList.remove('active'));
                tab.classList.add('active');
                this.renderTabContent(tab.dataset.tab, task, partSummaries);
            });
        });

        // 默认显示配件汇总
        this.renderTabContent('details', task, partSummaries);
    },

    /**
     * 渲染标签页内容
     */
    renderTabContent(tabName, task, details) {
        const container = document.getElementById('tab-content');

        switch (tabName) {
            case 'details':
                this.renderDetailsTab(container, details);
                break;

            case 'logs':
                this.renderLogsTab(container, this._currentLogs);
                break;
            case 'report':
                this.renderReportTab(container, task.task_no);
                break;
            case 'info':
                this.renderInfoTab(container, task);
                break;
        }

        lucide.createIcons();
    },

    /**
     * 加载配件汇总数据（支持排序和筛选）
     */
    async loadPartSummaries() {
        if (!this.currentTaskNo) return;
        
        try {
            const params = {
                page: this.partSummaryFilters.page,
                page_size: this.partSummaryFilters.page_size,
                sort_by: this.partSummarySort.sortBy,
                sort_order: this.partSummarySort.sortOrder,
                part_code: this.partSummaryFilters.part_code,
                priority: this.partSummaryFilters.priority
            };
            
            // 移除空值参数
            Object.keys(params).forEach(key => {
                if (params[key] === '' || params[key] === null || params[key] === undefined) {
                    delete params[key];
                }
            });

            const data = await API.getPartSummaries(this.currentTaskNo, params);
            this._partSummaries = data;
            
            const container = document.getElementById('tab-content');
            if (container) {
                this.renderDetailsTab(container, data);
                lucide.createIcons();
            }
        } catch (error) {
            Components.showToast('加载配件数据失败: ' + error.message, 'error');
        }
    },

    /**
     * 切换配件汇总排序
     */
    togglePartSummarySort(field) {
        if (this.partSummarySort.sortBy === field) {
            this.partSummarySort.sortOrder = this.partSummarySort.sortOrder === 'desc' ? 'asc' : 'desc';
        } else {
            this.partSummarySort.sortBy = field;
            this.partSummarySort.sortOrder = 'desc';
        }
        this.loadPartSummaries();
    },

    /**
     * 获取排序图标
     */
    getSortIcon(field) {
        if (this.partSummarySort.sortBy !== field) {
            return '<i data-lucide="arrow-up-down" class="sort-icon sort-icon-inactive"></i>';
        }
        if (this.partSummarySort.sortOrder === 'desc') {
            return '<i data-lucide="arrow-down" class="sort-icon sort-icon-active"></i>';
        }
        return '<i data-lucide="arrow-up" class="sort-icon sort-icon-active"></i>';
    },

    /**
     * 渲染配件明细标签页
     */
    renderDetailsTab(container, partSummaries) {
        const items = partSummaries.items || [];
        const { total, page, page_size } = partSummaries;

        container.innerHTML = `
            <div class="table-container">
                <div class="table-header" style="flex-wrap: wrap; gap: 1rem; height: auto;">
                    <div style="display: flex; align-items: center; gap: 1rem; flex: 1;">
                        <h3 class="table-title">配件补货建议 (商家组合维度) - ${total}个配件</h3>
                        <div class="table-header-hint">
                            <i data-lucide="info" style="width:14px;height:14px;"></i>
                            <span>点击表头可排序</span>
                        </div>
                    </div>
                    
                    <div class="filter-toolbar" style="display: flex; gap: 0.5rem; align-items: center;">
                        <input type="text" 
                            id="filter-part-code" 
                            class="input input-sm" 
                            placeholder="搜索配件编码..." 
                            value="${this.partSummaryFilters.part_code || ''}"
                            style="width: 150px;"
                        >
                        <select id="filter-priority" class="input input-sm" style="width: 120px;">
                            <option value="">所有优先级</option>
                            <option value="1" ${this.partSummaryFilters.priority == 1 ? 'selected' : ''}>急需补货</option>
                            <option value="2" ${this.partSummaryFilters.priority == 2 ? 'selected' : ''}>建议补货</option>
                            <option value="3" ${this.partSummaryFilters.priority == 3 ? 'selected' : ''}>可选补货</option>
                            <option value="0" ${this.partSummaryFilters.priority === '0' || this.partSummaryFilters.priority === 0 ? 'selected' : ''}>无需补货</option>
                        </select>
                        <button class="btn btn-secondary btn-sm" onclick="App.applyPartFilters()">
                            <i data-lucide="search" style="width: 14px; height: 14px; margin-right: 4px;"></i>
                            查询
                        </button>
                        <button class="btn btn-ghost btn-sm" onclick="App.resetPartFilters()">
                            重置
                        </button>
                    </div>
                </div>
                <div class="table-wrapper">
                    <table>
                        <thead>
                            <tr>
                                <th style="width: 40px;"></th>
                                <th class="sortable-th" onclick="App.togglePartSummarySort('part_code')">
                                    配件编码 ${this.getSortIcon('part_code')}
                                </th>
                                <th>配件名称</th>
                                <th class="sortable-th" onclick="App.togglePartSummarySort('cost_price')">
                                    成本价 ${this.getSortIcon('cost_price')}
                                </th>
                                <th class="sortable-th" onclick="App.togglePartSummarySort('total_storage_cnt')">
                                    总库存 ${this.getSortIcon('total_storage_cnt')}
                                </th>
                                <th class="sortable-th" onclick="App.togglePartSummarySort('total_avg_sales_cnt')">
                                    总销量 ${this.getSortIcon('total_avg_sales_cnt')}
                                </th>
                                <th class="sortable-th" onclick="App.togglePartSummarySort('group_current_ratio')">
                                    商家组合库销比 ${this.getSortIcon('group_current_ratio')}
                                </th>
                                <th class="sortable-th" onclick="App.togglePartSummarySort('group_post_plan_ratio')">
                                    计划后库销比 ${this.getSortIcon('group_post_plan_ratio')}
                                </th>
                                <th class="sortable-th" onclick="App.togglePartSummarySort('shop_count')">
                                    门店数 ${this.getSortIcon('shop_count')}
                                </th>
                                <th class="sortable-th" onclick="App.togglePartSummarySort('need_replenishment_shop_count')">
                                    需补货门店 ${this.getSortIcon('need_replenishment_shop_count')}
                                </th>
                                <th class="sortable-th" onclick="App.togglePartSummarySort('total_suggest_cnt')">
                                    总建议数量 ${this.getSortIcon('total_suggest_cnt')}
                                </th>
                                <th class="sortable-th" onclick="App.togglePartSummarySort('total_suggest_amount')">
                                    总建议金额 ${this.getSortIcon('total_suggest_amount')}
                                </th>
                            </tr>
                        </thead>
                        <tbody>
                            ${items.length > 0 ? items.map((item, index) => `
                                <tr class="part-summary-row" data-part-code="${item.part_code}" data-index="${index}">
                                    <td>
                                        <button class="btn btn-ghost btn-sm expand-btn" onclick="App.togglePartShops('${item.part_code}', ${index})">
                                            <i data-lucide="chevron-right" class="expand-icon"></i>
                                        </button>
                                    </td>
                                    <td class="table-cell-mono">${item.part_code}</td>
                                    <td>${item.part_name || '-'}</td>
                                    <td>${Components.formatAmount(item.cost_price)}</td>
                                    <td>${Components.formatNumber(item.total_storage_cnt)}</td>
                                    <td>${Components.formatNumber(item.total_avg_sales_cnt)}</td>
                                    <td>${Components.getRatioIndicator(item.group_current_ratio, 1.1)}</td>
                                    <td>${Components.formatRatio(item.group_post_plan_ratio)}</td>
                                    <td>${item.shop_count}</td>
                                    <td><strong style="color: var(--color-warning);">${item.need_replenishment_shop_count}</strong></td>
                                    <td><strong>${item.total_suggest_cnt}</strong></td>
                                    <td class="table-cell-amount">${Components.formatAmount(item.total_suggest_amount)}</td>
                                </tr>
                                <tr class="part-shops-row" id="shops-${index}" style="display: none;">
                                    <td colspan="12" style="padding: 0;">
                                        <div class="shops-container" id="shops-container-${index}">
                                            <div class="loading-shops">加载中...</div>
                                        </div>
                                    </td>
                                </tr>
                            `).join('') : `
                                <tr>
                                    <td colspan="12" style="text-align: center; padding: 2rem; color: var(--text-muted);">
                                        暂无符合条件的配件建议
                                    </td>
                                </tr>
                            `}
                        </tbody>
                    </table>
                </div>
                <div id="part-summary-pagination"></div>
            </div>

            <style>
                .part-summary-row { cursor: pointer; }
                .part-summary-row:hover { background: var(--bg-hover); }
                .expand-icon { transition: transform 0.2s; }
                .expand-icon.expanded { transform: rotate(90deg); }
                .shops-container { 
                    background: var(--bg-elevated); 
                    padding: var(--spacing-md); 
                    border-left: 3px solid var(--color-primary);
                    margin-left: var(--spacing-lg);
                }
                .loading-shops { 
                    color: var(--text-muted); 
                    padding: var(--spacing-sm); 
                }
                .shop-items-table {
                    width: 100%;
                    font-size: 0.875rem;
                }
                .shop-items-table th { 
                    background: var(--bg-subtle); 
                    font-weight: 600;
                    padding: var(--spacing-xs) var(--spacing-sm);
                }
                .shop-items-table td { 
                    padding: var(--spacing-xs) var(--spacing-sm); 
                }
                .part-decision-reason {
                    margin-bottom: var(--spacing-sm);
                    padding: var(--spacing-sm);
                    background: var(--bg-subtle);
                    border-radius: var(--radius-sm);
                    font-size: 0.875rem;
                    color: var(--text-secondary);
                }
            </style>
        `;

        // 渲染分页
        const paginationContainer = document.getElementById('part-summary-pagination');
        if (paginationContainer) {
            paginationContainer.innerHTML = Components.renderPagination(page, total, page_size);
            
            // 绑定分页事件
            paginationContainer.querySelectorAll('.pagination-btn[data-page]').forEach(btn => {
                btn.addEventListener('click', () => {
                    const targetPage = parseInt(btn.dataset.page);
                    if (targetPage && targetPage !== page) {
                        this.partSummaryFilters.page = targetPage;
                        this.loadPartSummaries();
                    }
                });
            });
        }
        
        // 绑定搜索框回车事件
        const partCodeInput = document.getElementById('filter-part-code');
        if (partCodeInput) {
            partCodeInput.addEventListener('keypress', (e) => {
                if (e.key === 'Enter') {
                    App.applyPartFilters();
                }
            });
        }
    },

    /**
     * 切换配件门店展开/收起
     */
    async togglePartShops(partCode, index) {
        const row = document.getElementById(`shops-${index}`);
        const container = document.getElementById(`shops-container-${index}`);
        const btn = document.querySelector(`tr[data-index="${index}"] .expand-icon`);
        
        if (row.style.display === 'none') {
            row.style.display = 'table-row';
            btn.classList.add('expanded');
            
            try {
                const data = await API.getPartShopDetails(this.currentTaskNo, partCode);
                const partSummary = this._partSummaries.items.find(p => p.part_code === partCode);
                this.renderPartShops(container, data.items, partSummary);
                lucide.createIcons();
            } catch (error) {
                container.innerHTML = `<div class="error-text">加载失败: ${error.message}</div>`;
            }
        } else {
            row.style.display = 'none';
            btn.classList.remove('expanded');
        }
    },

    /**
     * 渲染配件门店明细
     */
    renderPartShops(container, shops, partSummary) {
        if (!shops || shops.length === 0) {
            container.innerHTML = '<div class="text-muted">无门店建议数据</div>';
            return;
        }

        container.innerHTML = `
            ${partSummary && partSummary.part_decision_reason ? `
                <div class="part-decision-reason">
                    <strong><i data-lucide="message-square" style="width:14px;height:14px;"></i> 配件补货理由:</strong>
                    ${partSummary.part_decision_reason}
                </div>
            ` : ''}
            <table class="shop-items-table">
                <thead>
                    <tr>
                        <th>库房</th>
                        <th>有效库存</th>
                        <th>月均销量</th>
                        <th>当前库销比</th>
                        <th>计划后库销比</th>
                        <th>建议数量</th>
                        <th>建议金额</th>
                        <th>建议理由</th>
                    </tr>
                </thead>
                <tbody>
                    ${shops.map(shop => `
                        <tr>
                            <td>${shop.shop_name || '-'}</td>
                            <td>${Components.formatNumber(shop.valid_storage_cnt)}</td>
                            <td>${Components.formatNumber(shop.avg_sales_cnt)}</td>
                            <td>${Components.getRatioIndicator(shop.current_ratio, shop.base_ratio)}</td>
                            <td>${Components.formatRatio(shop.post_plan_ratio)}</td>
                            <td><strong>${shop.suggest_cnt}</strong></td>
                            <td class="table-cell-amount">${Components.formatAmount(shop.suggest_amount)}</td>
                            <td style="min-width: 200px;">
                                ${shop.suggestion_reason || '-'}
                            </td>
                        </tr>
                    `).join('')}
                </tbody>
            </table>
        `;
    },

    /**
     * 渲染执行日志标签页
     */
    renderLogsTab(container, logs) {
        if (!logs || !logs.items || logs.items.length === 0) {
            container.innerHTML = `
                <div class="card">
                    ${Components.renderEmptyState('activity', '暂无执行日志', '该任务没有执行日志记录')}
                </div>
            `;
            return;
        }

        const items = logs.items;
        const totalTokens = items.reduce((sum, item) => sum + (item.llm_tokens || 0), 0);
        const totalTime = items.reduce((sum, item) => sum + (item.execution_time_ms || 0), 0);

        container.innerHTML = `
            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">
                        <i data-lucide="activity"></i>
                        执行日志时间线
                    </h3>
                    <div class="card-actions">
                        <span class="text-muted">总耗时: ${Components.formatDuration(totalTime / 1000)} | 总Tokens: ${totalTokens}</span>
                    </div>
                </div>
                <div class="timeline">
                    ${items.map((log, index) => `
                        <div class="timeline-item ${log.status === 2 ? 'timeline-item-error' : 'timeline-item-success'}">
                            <div class="timeline-marker">
                                <div class="timeline-icon">
                                    ${log.status === 1 ? '<i data-lucide="check-circle"></i>' : 
                                      log.status === 2 ? '<i data-lucide="x-circle"></i>' : 
                                      '<i data-lucide="loader"></i>'}
                                </div>
                                ${index < items.length - 1 ? '<div class="timeline-line"></div>' : ''}
                            </div>
                            <div class="timeline-content">
                                <div class="timeline-header">
                                    <span class="timeline-title">${Components.getStepNameDisplay(log.step_name)}</span>
                                    ${Components.getLogStatusBadge(log.status)}
                                </div>
                                <div class="timeline-meta">
                                    <span class="meta-item">
                                        <i data-lucide="clock"></i>
                                        ${log.execution_time_ms ? Components.formatDuration(log.execution_time_ms / 1000) : '-'}
                                    </span>
                                    ${log.llm_tokens > 0 ? `
                                        <span class="meta-item">
                                            <i data-lucide="cpu"></i>
                                            ${log.llm_tokens} tokens
                                        </span>
                                    ` : ''}
                                    ${log.retry_count > 0 ? `
                                        <span class="meta-item meta-warning">
                                            <i data-lucide="refresh-cw"></i>
                                            重试 ${log.retry_count} 次
                                        </span>
                                    ` : ''}
                                </div>
                                ${log.error_message ? `
                                    <div class="timeline-error">
                                        <i data-lucide="alert-triangle"></i>
                                        ${log.error_message}
                                    </div>
                                ` : ''}
                            </div>
                        </div>
                    `).join('')}
                </div>
            </div>
        `;
    },

    /**
     * 渲染任务信息标签页
     */
    renderInfoTab(container, task) {
        container.innerHTML = `
            <div class="detail-grid">
                <div class="card">
                    <div class="card-header">
                        <h3 class="card-title">
                            <i data-lucide="info"></i>
                            基本信息
                        </h3>
                    </div>
                    <div class="info-list">
                        ${Components.renderInfoItem('任务编号', task.task_no)}
                        ${Components.renderInfoItem('集团ID', task.group_id)}
                        ${Components.renderInfoItem('商家组合ID', task.dealer_grouping_id)}
                        ${Components.renderInfoItem('商家组合名称', task.dealer_grouping_name)}
                        ${Components.renderInfoItem('品牌组合ID', task.brand_grouping_id)}
                        ${Components.renderInfoItem('统计日期', task.statistics_date)}
                    </div>
                </div>
                
                <div class="card">
                    <div class="card-header">
                        <h3 class="card-title">
                            <i data-lucide="activity"></i>
                            执行信息
                        </h3>
                    </div>
                    <div class="info-list">
                        ${Components.renderInfoItem('状态', Components.getStatusBadge(task.status, task.status_text))}
                        ${Components.renderInfoItem('开始时间', task.start_time)}
                        ${Components.renderInfoItem('结束时间', task.end_time)}
                        ${Components.renderInfoItem('执行时长', Components.formatDuration(task.duration_seconds))}
                        ${Components.renderInfoItem('创建时间', task.create_time)}
                    </div>
                </div>
                
                <div class="card">
                    <div class="card-header">
                        <h3 class="card-title">
                            <i data-lucide="cpu"></i>
                            LLM 信息
                        </h3>
                    </div>
                    <div class="info-list">
                        ${Components.renderInfoItem('LLM 提供商', task.llm_provider || '-')}
                        ${Components.renderInfoItem('模型名称', task.llm_model || '-')}
                        ${Components.renderInfoItem('Token 消耗', task.llm_total_tokens)}
                    </div>
                </div>
                
                ${task.error_message ? `
                <div class="card">
                    <div class="card-header">
                        <h3 class="card-title" style="color: var(--color-danger)">
                            <i data-lucide="alert-triangle"></i>
                            错误信息
                        </h3>
                    </div>
                    <pre style="background: var(--bg-elevated); padding: var(--spacing-md); border-radius: var(--radius-md); overflow-x: auto; color: var(--color-danger-light);">${task.error_message}</pre>
                </div>
                ` : ''}
            </div>
        `;
    },
};

// DOM 加载完成后初始化
document.addEventListener('DOMContentLoaded', () => {
    App.init();
});

// 导出到全局
window.App = App;
