/**
 * API 调用封装
 */

const API = {
    baseUrl: '/api',

    /**
     * 通用请求方法
     */
    async request(endpoint, options = {}) {
        const url = `${this.baseUrl}${endpoint}`;
        
        try {
            const response = await fetch(url, {
                ...options,
                headers: {
                    'Content-Type': 'application/json',
                    ...options.headers,
                },
            });

            if (!response.ok) {
                const error = await response.json().catch(() => ({}));
                throw new Error(error.detail || `请求失败: ${response.status}`);
            }

            return await response.json();
        } catch (error) {
            console.error('API Error:', error);
            throw error;
        }
    },

    /**
     * GET 请求
     */
    async get(endpoint, params = {}) {
        const queryString = new URLSearchParams(params).toString();
        const url = queryString ? `${endpoint}?${queryString}` : endpoint;
        return this.request(url);
    },

    /**
     * 获取任务列表
     */
    async getTasks(params = {}) {
        return this.get('/tasks', params);
    },

    /**
     * 获取任务详情
     */
    async getTask(taskNo) {
        return this.get(`/tasks/${taskNo}`);
    },

    /**
     * 获取任务配件明细
     */
    async getTaskDetails(taskNo, params = {}) {
        return this.get(`/tasks/${taskNo}/details`, params);
    },


    /**
     * 获取统计摘要
     */
    async getStatsSummary() {
        return this.get('/stats/summary');
    },

    /**
     * 获取任务执行日志
     */
    async getTaskLogs(taskNo) {
        return this.get(`/tasks/${taskNo}/logs`);
    },

    /**
     * 获取任务配件汇总列表
     */
    async getPartSummaries(taskNo, params = {}) {
        return this.get(`/tasks/${taskNo}/part-summaries`, params);
    },

    /**
     * 获取配件的门店明细
     */
    async getPartShopDetails(taskNo, partCode) {
        return this.get(`/tasks/${taskNo}/parts/${encodeURIComponent(partCode)}/shops`);
    },

    /**
     * 获取任务分析报告
     */
    async getAnalysisReport(taskNo) {
        return this.get(`/tasks/${taskNo}/analysis-report`);
    },
};

// 导出到全局
window.API = API;
