"""
配置管理模块
使用 pydantic-settings 从环境变量加载配置
"""

from functools import lru_cache
from pydantic_settings import BaseSettings, SettingsConfigDict


class Settings(BaseSettings):
    """应用配置"""

    model_config = SettingsConfigDict(
        env_file=".env",
        env_file_encoding="utf-8",
        case_sensitive=False,
    )

    # GLM-4.7 配置
    glm_api_key: str = ""
    glm_model: str = "glm-4"

    # OpenAI 兼容模式配置（火山引擎等）
    openai_compat_api_key: str = ""
    openai_compat_base_url: str = "https://ark.cn-beijing.volces.com/api/v3"
    openai_compat_model: str = "doubao-seed-1-8-251228"

    # Anthropic 兼容模式配置（智谱AI）
    anthropic_api_key: str = ""
    anthropic_base_url: str = "https://open.bigmodel.cn/api/anthropic"
    anthropic_model: str = "glm-4.7"

    # 豆包配置
    doubao_api_key: str = ""
    doubao_model: str = "doubao-pro"

    # 数据库配置
    mysql_host: str = "localhost"
    mysql_port: int = 3306
    mysql_user: str = "root"
    mysql_password: str = ""
    mysql_database: str = "fw_pms"

    # 定时任务配置
    scheduler_cron_hour: int = 2
    scheduler_cron_minute: int = 0

    # 日志配置
    log_level: str = "INFO"

    @property
    def mysql_connection_string(self) -> str:
        """MySQL 连接字符串"""
        return (
            f"mysql+mysqlconnector://{self.mysql_user}:{self.mysql_password}"
            f"@{self.mysql_host}:{self.mysql_port}/{self.mysql_database}"
        )

    @property
    def primary_llm_provider(self) -> str:
        """主要 LLM 供应商"""
        if self.openai_compat_api_key:
            return "openai_compat"
        elif self.anthropic_api_key:
            return "anthropic_compat"
        elif self.glm_api_key:
            return "glm"
        elif self.doubao_api_key:
            return "doubao"
        else:
            raise ValueError("未配置任何 LLM API Key")


@lru_cache
def get_settings() -> Settings:
    """获取配置单例"""
    return Settings()
