"""
FastAPI 主应用
提供 AI 补货建议系统的 REST API
"""

import logging
from pathlib import Path
from contextlib import asynccontextmanager

from fastapi import FastAPI
from fastapi.middleware.cors import CORSMiddleware
from fastapi.staticfiles import StaticFiles
from fastapi.responses import FileResponse

from .routes import tasks

logger = logging.getLogger(__name__)


@asynccontextmanager
async def lifespan(app: FastAPI):
    """应用生命周期管理"""
    logger.info("API 服务启动")
    yield
    logger.info("API 服务关闭")


app = FastAPI(
    title="AI 补货建议系统 API",
    description="提供补货任务、明细的查询接口",
    version="1.0.0",
    lifespan=lifespan,
)

# CORS 配置
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

# 挂载路由
app.include_router(tasks.router, prefix="/api", tags=["Tasks"])

# 静态文件服务
ui_path = Path(__file__).parent.parent.parent.parent / "ui"
if ui_path.exists():
    app.mount("/css", StaticFiles(directory=ui_path / "css"), name="css")
    app.mount("/js", StaticFiles(directory=ui_path / "js"), name="js")


@app.get("/", include_in_schema=False)
async def serve_index():
    """服务主页面"""
    index_file = ui_path / "index.html"
    if index_file.exists():
        return FileResponse(index_file)
    return {"message": "AI 补货建议系统 API", "docs": "/docs"}


@app.get("/health")
async def health_check():
    """健康检查"""
    return {"status": "ok"}
