-- ============================================================================
-- AI 补货建议系统 - 表结构
-- ============================================================================
-- 版本: 1.0.0
-- 更新日期: 2026-01-31
-- ============================================================================

-- 1. AI补货任务表
DROP TABLE IF EXISTS ai_replenishment_task;
CREATE TABLE ai_replenishment_task (
    id                   BIGINT AUTO_INCREMENT PRIMARY KEY COMMENT '主键ID',
    task_no              VARCHAR(32) NOT NULL UNIQUE COMMENT '任务编号(AI-开头)',
    group_id             BIGINT NOT NULL COMMENT '集团ID',
    dealer_grouping_id   BIGINT NOT NULL COMMENT '商家组合ID',
    dealer_grouping_name VARCHAR(128) COMMENT '商家组合名称',
    brand_grouping_id    BIGINT COMMENT '品牌组合ID',
    plan_amount          DECIMAL(14,2) DEFAULT 0 COMMENT '计划采购金额(预算)',
    actual_amount        DECIMAL(14,2) DEFAULT 0 COMMENT '实际分配金额',
    part_count           INT DEFAULT 0 COMMENT '分配配件数量',
    base_ratio           DECIMAL(10,4) COMMENT '基准库销比',
    status               TINYINT DEFAULT 0 COMMENT '状态: 0-运行中 1-成功 2-失败',
    error_message        TEXT COMMENT '错误信息',
    llm_provider         VARCHAR(32) COMMENT 'LLM提供商',
    llm_model            VARCHAR(64) COMMENT 'LLM模型名称',
    llm_total_tokens     INT DEFAULT 0 COMMENT 'LLM总Token数',
    statistics_date      VARCHAR(16) COMMENT '统计日期',
    start_time           DATETIME COMMENT '任务开始时间',
    end_time             DATETIME COMMENT '任务结束时间',
    create_time          DATETIME DEFAULT CURRENT_TIMESTAMP COMMENT '创建时间',
    INDEX idx_group_date (group_id, statistics_date),
    INDEX idx_dealer_grouping (dealer_grouping_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='AI补货任务表-记录每次补货建议任务执行';

-- 2. AI补货建议明细表
DROP TABLE IF EXISTS ai_replenishment_detail;
CREATE TABLE ai_replenishment_detail (
    id                   BIGINT AUTO_INCREMENT PRIMARY KEY COMMENT '主键ID',
    task_no              VARCHAR(32) NOT NULL COMMENT '任务编号',
    group_id             BIGINT NOT NULL COMMENT '集团ID',
    dealer_grouping_id   BIGINT NOT NULL COMMENT '商家组合ID',
    brand_grouping_id    BIGINT COMMENT '品牌组合ID',
    shop_id              BIGINT NOT NULL COMMENT '库房ID',
    shop_name            VARCHAR(128) COMMENT '库房名称',
    part_code            VARCHAR(64) NOT NULL COMMENT '配件编码',
    part_name            VARCHAR(256) COMMENT '配件名称',
    unit                 VARCHAR(32) COMMENT '单位',
    cost_price           DECIMAL(14,2) DEFAULT 0 COMMENT '成本价',
    current_ratio        DECIMAL(10,4) COMMENT '当前库销比',
    base_ratio           DECIMAL(10,4) COMMENT '基准库销比',
    post_plan_ratio      DECIMAL(10,4) COMMENT '计划后预计库销比',
    valid_storage_cnt    DECIMAL(14,2) DEFAULT 0 COMMENT '有效库存数量',
    avg_sales_cnt        DECIMAL(14,2) DEFAULT 0 COMMENT '平均销量(月)',
    suggest_cnt          INT DEFAULT 0 COMMENT '建议采购数量',
    suggest_amount       DECIMAL(14,2) DEFAULT 0 COMMENT '建议采购金额',
    suggestion_reason    TEXT COMMENT '补货建议理由',
    priority             INT NOT NULL DEFAULT 2 COMMENT '优先级: 1-高, 2-中, 3-低',
    llm_confidence       FLOAT DEFAULT 0.8 COMMENT 'LLM置信度',
    statistics_date      VARCHAR(16) COMMENT '统计日期',
    create_time          DATETIME DEFAULT CURRENT_TIMESTAMP COMMENT '创建时间',
    INDEX idx_task_no (task_no),
    INDEX idx_shop_part (shop_id, part_code)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='AI补货建议明细表-存储每次任务的配件分配结果';

-- 3. AI任务执行日志表
DROP TABLE IF EXISTS ai_task_execution_log;
CREATE TABLE ai_task_execution_log (
    id                   BIGINT AUTO_INCREMENT PRIMARY KEY COMMENT '主键ID',
    task_no              VARCHAR(32) NOT NULL COMMENT '任务编号',
    group_id             BIGINT NOT NULL COMMENT '集团ID',
    brand_grouping_id    BIGINT COMMENT '品牌组合ID',
    brand_grouping_name  VARCHAR(128) COMMENT '品牌组合名称',
    dealer_grouping_id   BIGINT NOT NULL COMMENT '商家组合ID',
    dealer_grouping_name VARCHAR(128) COMMENT '商家组合名称',
    step_name            VARCHAR(64) NOT NULL COMMENT '步骤名称',
    step_order           INT DEFAULT 0 COMMENT '步骤顺序',
    status               TINYINT DEFAULT 0 COMMENT '状态: 0-进行中 1-成功 2-失败 3-跳过',
    input_data           TEXT COMMENT '输入数据(JSON)',
    output_data          TEXT COMMENT '输出数据(JSON)',
    error_message        TEXT COMMENT '错误信息',
    retry_count          INT DEFAULT 0 COMMENT '重试次数',
    sql_query            TEXT COMMENT 'SQL查询语句(如有)',
    llm_prompt           TEXT COMMENT 'LLM提示词(如有)',
    llm_response         TEXT COMMENT 'LLM响应(如有)',
    llm_tokens           INT DEFAULT 0 COMMENT 'LLM Token消耗',
    execution_time_ms    INT DEFAULT 0 COMMENT '执行耗时(毫秒)',
    start_time           DATETIME COMMENT '开始时间',
    end_time             DATETIME COMMENT '结束时间',
    create_time          DATETIME DEFAULT CURRENT_TIMESTAMP COMMENT '创建时间',
    INDEX idx_task_no (task_no),
    INDEX idx_group_date (group_id, create_time),
    INDEX idx_dealer_grouping (dealer_grouping_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='AI任务执行日志表-记录每个步骤的执行详情';

-- 4. AI补货配件汇总表
DROP TABLE IF EXISTS ai_replenishment_part_summary;
CREATE TABLE ai_replenishment_part_summary (
    id                            BIGINT AUTO_INCREMENT PRIMARY KEY COMMENT '主键ID',
    task_no                       VARCHAR(32) NOT NULL COMMENT '任务编号',
    group_id                      BIGINT NOT NULL COMMENT '集团ID',
    dealer_grouping_id            BIGINT NOT NULL COMMENT '商家组合ID',
    part_code                     VARCHAR(64) NOT NULL COMMENT '配件编码',
    part_name                     VARCHAR(256) COMMENT '配件名称',
    unit                          VARCHAR(32) COMMENT '单位',
    cost_price                    DECIMAL(14,2) DEFAULT 0.00 COMMENT '成本价',
    total_storage_cnt             DECIMAL(14,2) DEFAULT 0.00 COMMENT '商家组合内总库存数量',
    total_avg_sales_cnt           DECIMAL(14,2) DEFAULT 0.00 COMMENT '商家组合内总月均销量',
    group_current_ratio           DECIMAL(10,4) COMMENT '商家组合级库销比',
    total_suggest_cnt             INT DEFAULT 0 COMMENT '总建议数量',
    total_suggest_amount          DECIMAL(14,2) DEFAULT 0.00 COMMENT '总建议金额',
    shop_count                    INT DEFAULT 0 COMMENT '涉及门店数',
    need_replenishment_shop_count INT DEFAULT 0 COMMENT '需要补货的门店数',
    part_decision_reason          TEXT COMMENT '配件级补货决策理由',
    priority                      INT NOT NULL DEFAULT 2 COMMENT '优先级: 1=高, 2=中, 3=低',
    llm_confidence                FLOAT DEFAULT 0.8 COMMENT 'LLM置信度',
    statistics_date               VARCHAR(16) COMMENT '统计日期',
    create_time                   DATETIME DEFAULT CURRENT_TIMESTAMP COMMENT '创建时间',
    INDEX idx_task_no (task_no),
    INDEX idx_part_code (part_code),
    INDEX idx_dealer_grouping (dealer_grouping_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='AI补货建议-配件汇总表';

